/*
*
* Copyright (c) 2006 Mellanox Technologies. All rights reserved.
*
* This Software is licensed under one of the following licenses:
*
* 1) under the terms of the "Common Public License 1.0" a copy of which is
*    available from the Open Source Initiative, see
*    http://www.opensource.org/licenses/cpl.php.
*
* 2) under the terms of the "The BSD License" a copy of which is
*    available from the Open Source Initiative, see
*    http://www.opensource.org/licenses/bsd-license.php.
*
* 3) under the terms of the "GNU General Public License (GPL) Version 2" a
*    copy of which is available from the Open Source Initiative, see
*    http://www.opensource.org/licenses/gpl-license.php.
*
* Licensee has the right to choose one of the above licenses.
*
* Redistributions of source code must retain the above copyright
* notice and one of the license notices.
*
* Redistributions in binary form must reproduce both the above copyright
* notice, one of the license notices in the documentation
* and/or other materials provided with the distribution.
*
*  $Id$
*
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <sys/errno.h>
#include <net/if.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>



#include "list.h"

#define TEST_PORT   3456
#ifndef INVALID_SOCKET
#  define INVALID_SOCKET -1
#endif /* INVALID_SOCKET */
#ifndef SOCKET_ERROR
#  define SOCKET_ERROR -1
#endif /* SOCKET_ERROR */

#ifndef INET6_ADDRSTRLEN
#define INET6_ADDRSTRLEN (46)
#endif

typedef int s_sock_t;
typedef socklen_t s_socklen_t;

sig_atomic_t sig_count = 0;
struct addr_entry  entry;
struct addr_entry* iter; 

extern int sched_yield(void);

void handler(int signal_number)
{
    if ( signal_number == SIGTERM ) {
        ++sig_count;
    }
}

struct addr_entry
{
    struct list_head list;
    char local_ip[16];
    char mcast_ip[256];
    unsigned int ifindex;
    s_sock_t s;
};

static void append(struct addr_entry* ptr,const char* local_ip, const char* mcast_ip, const s_sock_t s, const unsigned int ifindex)
{
  struct addr_entry* tmp;
  tmp = (struct addr_entry*)malloc(sizeof(struct addr_entry));
  if(!tmp) {
    perror("malloc");
    exit(1);
  }
  strcpy(tmp->local_ip, local_ip);
  strcpy(tmp->mcast_ip, mcast_ip);
  tmp->ifindex = ifindex;
  tmp->s = s;
  list_add_tail( &(tmp->list), &(ptr->list) );
}

int join_mcast(const char* local_ip, const char* mcast_ip, const unsigned int ifindex)
{
   s_sock_t s;
   struct sockaddr_in stLocal;
   struct ip_mreq stMreq;
   int iTmp, iRet;

   /* name the socket */
   stLocal.sin_family = AF_INET;

   /* get a datagram socket */
   s = socket(AF_INET, SOCK_DGRAM, 0);
   if (s == INVALID_SOCKET) {
     printf ("socket() failed, Err: %d\n", errno);
     exit (1);
   }

   /* avoid EADDRINUSE error on bind() */
   iTmp = 1;
   iRet = setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&iTmp, sizeof(iTmp));
   if (iRet == SOCKET_ERROR) {
      printf ("setsockopt() SO_REUSEADDR failed, Err: %d\n", errno);
   }
        
   stLocal.sin_addr.s_addr = htonl(INADDR_ANY);
//   stLocal.sin_port = htons(TEST_PORT);
   iRet = bind(s, (struct sockaddr *)&stLocal, sizeof(stLocal));
   if (iRet == SOCKET_ERROR) {
      printf ("bind() failed, Err: %d\n", errno);
   }

   /* join the multicast group */
   // printf("Mcast_ip=%s, Local_ip=%s\n", mcast_ip, local_ip);
   stMreq.imr_multiaddr.s_addr = inet_addr(mcast_ip);
   stMreq.imr_interface.s_addr = inet_addr(local_ip);

   iRet = setsockopt(s, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *)&stMreq, sizeof(stMreq));
   if (iRet == SOCKET_ERROR) {
     printf ("setsockopt() IP_ADD_MEMBERSHIP failed, Err: %d\n", errno);
   }

  append(&entry, local_ip, mcast_ip, s, ifindex);
  return 0;
}

int join_mcast6(const char* local_ip, const char* mcast_ip, const unsigned int ifindex)
{
    s_sock_t s;
    struct ipv6_mreq stMreq;;
    int iTmp, iRet;
    
    /* get a datagram socket */
    s = socket(AF_INET6, SOCK_DGRAM, 0);
    if (s == INVALID_SOCKET) {
        printf ("socket() failed, Err: %d\n", errno);
        exit (1);
    }
    
    /* avoid EADDRINUSE error on bind() */
    iTmp = 1;
    iRet = setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&iTmp, sizeof(iTmp));
    if (iRet == SOCKET_ERROR) {
        printf ("setsockopt() SO_REUSEADDR failed, Err: %d\n", errno);
    }
         
   
    if (inet_pton(AF_INET6, mcast_ip, &stMreq.ipv6mr_multiaddr.s6_addr) != 1)
    {
    	fprintf(stderr, "Failed inet_pton: %d\n", errno);
    }
  

    iRet = 0;
	stMreq.ipv6mr_interface = ifindex;
        
	if (setsockopt
			(s, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, (char *) &stMreq,
			 sizeof (stMreq)) < 0)
	{
		perror ("Can't join IPv6 multicast group");
        iRet = 1;
	}

    append(&entry, local_ip, mcast_ip, s, ifindex);
    return iRet;
}

int leave_mcast( const char* local_ip, const char* mcast_ip, s_sock_t s )
{
   int iRet;
   struct ip_mreq stMreq;


   /* leave the multicast group */
   stMreq.imr_multiaddr.s_addr = inet_addr(mcast_ip);
   stMreq.imr_interface.s_addr = inet_addr(local_ip);

   iRet = setsockopt(s, IPPROTO_IP, IP_DROP_MEMBERSHIP, (char *)&stMreq, sizeof(stMreq));
   if (iRet == SOCKET_ERROR) {
     printf ("setsockopt() IP_DROP_MEMBERSHIP failed, Err: %d\n", errno);
   }

   return iRet;
}

int leave_mcast6( const char* local_ip, const char* mcast_ip, s_sock_t s, const unsigned int ifindex )
{
    int iRet;
    struct ipv6_mreq stMreq;


    /* leave the multicast group */
    if (inet_pton(AF_INET6, mcast_ip, &stMreq.ipv6mr_multiaddr.s6_addr) != 1)
    {
    	fprintf(stderr, "Failed inet_pton: %d\n", errno);
    }

    stMreq.ipv6mr_interface = ifindex;

    iRet = setsockopt(s, IPPROTO_IPV6, IPV6_DROP_MEMBERSHIP, (char *)&stMreq, sizeof(stMreq));
    if (iRet == SOCKET_ERROR) {
        printf ("setsockopt() IPV6_DROP_MEMBERSHIP failed, Err: %d\n", errno);
    }

    return iRet;
}

int main(int argc, char *argv[])
{
    char act_file[50];
    char action[10];
    char addr_family[5];
    char local_ip[16];
    char mcast_ip[256];
    char buf[256];
    unsigned int ifindex = 0;
    FILE *fp;
    s_sock_t s = 0;

    INIT_LIST_HEAD(&entry.list);    
    
    if (argc != 2) {
      printf ("Usage: mcasthandle <file name>\n");
      exit (1);
    }
    strcpy (act_file, argv[1]);

    printf("File: %s\n", act_file); 
    while ( ! ( fp = fopen(act_file, "r") )) {
        sched_yield(); 
        usleep(100);
    }


    struct sigaction sa;
    memset (&sa, 0, sizeof(sa));
    sa.sa_handler = &handler;
    
    sigaction (SIGTERM, &sa, NULL);
    
    for( ; ; ) {
        while (fgets(buf, sizeof(buf), fp)) {
            struct addr_entry m;
            memset(&m, 0, sizeof(m));
            sscanf(buf, "%s%s%s%s%u", action, addr_family, local_ip, mcast_ip, &ifindex);
            // printf("action=%s addr_family=%s local_ip=%s mcast_ip=%s ifindex=%u\n", action, addr_family, local_ip, mcast_ip, ifindex);
            if ( strcmp( action, "add") == 0 ) {
                if ( strcmp( addr_family, "inet") == 0 )
                    join_mcast(local_ip, mcast_ip, ifindex);
                else
                    join_mcast6(local_ip, mcast_ip, ifindex);
            }
            else {
                // Remove entry from the list
                list_for_each_entry(iter,&entry.list,list) {
                    if ( ( strcmp (iter->local_ip, local_ip) == 0 ) &&
                        ( strcmp (iter->mcast_ip, mcast_ip) == 0 ) ) {
                            s = iter->s;
                            ifindex = iter->ifindex;
                            list_del(&iter->list);
                            break;
                    }
                }
                // Leave the multicast group
                if ( s ) {
                    if ( strcmp( addr_family, "inet") == 0 )
                        leave_mcast(local_ip, mcast_ip, s);
                    else
                        leave_mcast6(local_ip, mcast_ip, s, ifindex);
                }
            }

        }
        if (sig_count) {
            /* remove all items in the list */
            while( !list_empty(&entry.list) ) {
                iter = list_entry(entry.list.next,struct addr_entry,list);
                list_del(&iter->list);
                free(iter);
            }

            printf("Got SIGTERM. Exiting ...\n");
            break;
        }
        sched_yield(); 
        usleep(100);
   }

   fclose(fp);
   return 0;
} /* end main() */

